/**
 * Composition Profiler (relative entropy calculation)
 * 
 * Vladimir Vacic, University of California, Riverside
 * Vladimir N. Uversky, Indiana University School of Medicine, Indianapolis 
 * A. Keith Dunker, Indiana University School of Medicine, Indianapolis
 * Stefano Lonardi, University of California, Riverside
 *
 * Sep-27-2006
 */


#include <ctime>
#include <ctype.h>
#include <cstdlib>
#include <iostream>
#include <fstream>
#include <math.h>
#include <string>
#include <vector>
using namespace std;

#include "sequence.h"
#define LN_2 0.69314718056



/** Relative entropy between two vectors of probabilities */
double rentropy(const vector<double> &p, const vector<double> &q)  {
    double r(0);    
    for (int i=0; i<p.size(); i++)
	if (p[i]!=0 && q[i]!=0)
	    r += p[i] * log(p[i]/q[i]) / LN_2; 

    return r;
}

int main(int argc, char** argv)  {
    if (argc != 4)  {
        cerr << "Usage: " << argv[0] << " P_sample Q_sample bootstrap_iterations\n";
        exit(1);
    } 

    // Count AA in both samples
    vector<vector<int> > pcounts = count_aa(argv[1]);
    vector<vector<int> > qcounts = count_aa(argv[2]);
    int pseqnum = pcounts.size();
    int qseqnum = qcounts.size();

    int iterations = atoi(argv[3]);

    vector<int> pindex(pseqnum, 0);
    vector<int> qindex(qseqnum, 0); 

    // Observed relative entropy between P and Q
    for (int i=1; i<pseqnum; i++)  pindex[i] = i;
    for (int i=1; i<qseqnum; i++)  qindex[i] = i;

    double observed = rentropy(freq_aa(pindex, pcounts), freq_aa(qindex, qcounts));

    double count(0);
    double ptemp = (double) pseqnum / (1.0 + RAND_MAX);
    double qtemp = (double) qseqnum / (1.0 + RAND_MAX);

    srand((unsigned)time(0)); 

    for (int i=0; i<iterations; i++)  {
        // Resample P on the sequence level
        for (int j=0; j<pseqnum; j++)
            // Random sample of integers from 0 to n-1, with replacement
            pindex[j] = (int) floor(ptemp * rand());

        // Resample Q on the sequence level
        for (int j=0; j<qseqnum; j++)
            qindex[j] = (int) floor(qtemp * rand());

	// Compare the bootstrap entropy to the observed entropy
	if (rentropy(freq_aa(pindex, pcounts), freq_aa(qindex, qcounts)) >= observed)
            count++;
    }

    cout << observed << endl << count / iterations << endl;     
    exit(0);
}

