#
# Composition Profiler
#
# DrawChart.rb - fills in the EPS template file with data, and
#               creates the EPS file. This file can be further
#               converted into PDF, GIF or PNG formats. 
#
# Vladimir Vacic, University of California, Riverside
# Vladimir N. Uversky, Indiana University School of Medicine, Indianapolis
# A. Keith Dunker, Indiana University School of Medicine, Indianapolis
# Stefano Lonardi, University of California, Riverside
#
# Oct-21-2006


require "./ColorScheme.rb"
include ColorScheme

class DrawChart 
    #
    # Creates the EPS file accoring to the data and a set of parameters,
    # and if the output format is different from EPS, converts it into
    # GIF, PNG, or PDF.
    # 
    # Input: labels  - x-axis labels (amino acids sorted according to a
    #                  user-specified property)
    #        bars    - values of data points
    #        errors  - one standard deviation error bars
    #        options - hash of parameters passed from the driver script 
    #        outfile - the name of the output file
    #        path    - output directory (write permissions should be set 
    #                  accordingly) 
    #
    def DrawChart.create(labels, bars, errors, options, outfile, path)
        width  = options['WIDTH']
        height = options['HEIGHT']    
        res    = options['RES']

        if options['RES_UNITS']=="ppc"
            res *= 2.54
        elsif options['RES_UNITS']=="ppp"
            res *= 72 
        end

        format    = options['FORMAT']
        #antialias = options['ANTIALIAS'] ? "-dTextAlphaBits=4" : ""
	antialias = ((options['ANTIALIAS'] and format != "PDF") ? "-dTextAlphaBits=4" : "")

        output = rand.to_s[2,15].to_s

        temp = File.open("#{path}#{output}.dat", "w")
        for i in 0...20
            if bars[i]>0
                temp.print "#{i+1}\t#{bars[i]}\t#{bars[i]}\t#{bars[i]+errors[i]}\n"
            else
                temp.print "#{i+1}\t#{bars[i]}\t#{bars[i]+errors[i]}\t#{bars[i]}\n"
            end
        end
        temp.close


        temp = File.open("#{path}#{output}.p", "w")
        temp.print "set terminal postscript eps enhanced color \"Helvetica\" 20\n"
        temp.print "set size #{width/360.0},#{height/252.0}\n"

        temp.print "set tics out\n"
        temp.print "set xtics (" 
        for i in 0...labels.length
            if i==labels.length-1
                temp.print "\"#{labels[i]}\" #{i+1})\n"
            else
                temp.print "\"#{labels[i]}\" #{i+1},"
            end
        end

        temp.print "set xrange [0:21]\n"
        temp.print "set xzeroaxis lt -1\n"
        temp.print "set border 3\n"
        temp.print "set xtics nomirror\n"
        temp.print "set ytics nomirror\n"
               
        if nil!=options['YAXIS'] 
            temp.print "set ylabel \"#{options['YAXIS']}\"\n"
        end 

        temp.print "plot \"#{path}#{output}.dat\" using 1:2:(0.75) notitle with boxes fs solid 0.7,\\\n"
        temp.print "\"#{path}#{output}.dat\" using 1:2:3:4 notitle with yerrorbars lt -1 pt 9 ps 0.75\n"
        temp.close

        out = `gnuplot #{path}#{output}.p`

        eps = String.new 
        i = 0

        # hacking gnuplot-generated EPS
        out.each_line do |line|
#             if nil!=line.index("BoxColFill\n")
             if nil!=line.index("BoxColFill\n")

                 #CO: gnuplot adds a color to the first BoxColFill, strip it
                 color = line.index(" C ")
                 if nil!=color
                     line = line[(color+3)..-1]
                 end
 
                 if "bw"==options['COLOR_SCHEME']
                     eps << "0.5 0.5 0.5 setrgbcolor\n"
                 else
                     eps << COLOR_SCHEME[options['COLOR_SCHEME']][labels[i]] << " setrgbcolor\n"
                     i += 1
                 end
             end

             eps << line
        end

        File.delete("#{path}#{output}.dat")
        File.delete("#{path}#{output}.p")

        # GhostView and ImageMagick paths
        answer  = get_configuration()
        gs      = answer[0]
        convert = answer[1]
     
        if format == "EPS"
            if outfile == "-"  # if standard out
                puts eps
            else
                file = File.open("#{path}#{outfile}", "w")
                file << eps
                file.close
            end

        elsif format == "PDF"
            file = File.open("#{path}#{output}.eps", "w")
            file << eps
            file.close
            system "sed 's/50 50 translate//;s/%%BoundingBox:.*$/%%BoundingBox: 0 0 #{width} #{height}/' #{path}#{output}.eps | #{gs} -sOutputFile=#{path}#{outfile} -sDEVICE=pdfwrite -dPDFSETTINGS=/printer -q -r#{res} -dDEVICEWIDTHPOINTS=#{width} -dDEVICEHEIGHTPOINTS=#{height} -dEmbedAllFonts=true #{antialias} -dSAFER -dBATCH  -dNOPAUSE -_"
            File.delete("#{path}#{output}.eps")

        elsif format == "PNG"
            file = File.open("#{path}#{output}.eps", "w")
            file << eps
            file.close

            system "sed 's/50 50 translate//;s/%%BoundingBox:.*$/%%BoundingBox: 0 0 #{width} #{height}/' #{path}#{output}.eps | #{gs} -sOutputFile=#{path}#{outfile} -sDEVICE=png16m -q -r#{res} -dDEVICEWIDTHPOINTS=#{width} -dDEVICEHEIGHTPOINTS=#{height} #{antialias} -dSAFER -dBATCH -dNOPAUSE -_"
            File.delete("#{path}#{output}.eps")

        elsif format == "GIF"  # convert to PNG first, then GIF
            file = File.open("#{path}#{output}.eps", "w")
            file << eps
            file.close
            program = "sed 's/50 50 translate//;s/%%BoundingBox:.*$/%%BoundingBox: 0 0 #{width} #{height}/' #{path}#{output}.eps | #{gs} -sOutputFile=- -sDEVICE=png16m -q -r#{res} -dDEVICEWIDTHPOINTS=#{width} -dDEVICEHEIGHTPOINTS=#{height} #{antialias} -dSAFER -dBATCH -dNOPAUSE -_"

            if outfile == "-"
                program += " | #{convert} png:- gif:-"
            else
                program += " | #{convert} png:- #{path}#{outfile}"
            end

            system program

            File.delete("#{path}#{output}.eps")
        end
    end


    # Reads the configuration file (cprof.conf).
    #
    # Output: answer[0] - path to GhostScript binary gs
    #         answer[1] - path to ImageMagick binary convert
    def DrawChart.get_configuration
        answer = Array.new
        answer << "gs" << "convert"     

        # No configuration file, then return defaults.
        return answer if !FileTest.exist?("cprof.conf")

        config = File.open("cprof.conf")
    
        config.each do |line|
            next if line =~ /^\#/       # skip lines beginning with "#"

            if line =~ /^gs/i           # if looks like gs (case insensitive)
                line =~ /^\S+\=(.+)$/
                answer[0] = $1
            elsif line =~ /^convert/i   # if looks like convert (case insensitive)
                line =~ /^\S+\=(.+)$/
                answer[1] = $1
            end
        end

        if answer[0] == nil || !FileTest.exist?(answer[0])
            puts "Please check cprof.conf: gs program (#{answer[0]}) does not exist."
            exit
        end

        if answer[1] == nil || !FileTest.exist?(answer[1])
            puts "Please check cprof.conf: convert program (#{answer[1]}) does not exist." 
            exit
        end

        return answer
    end


end
